import { useState } from 'react'
import { useNavigate } from 'react-router-dom'
import { Calendar, CheckCircle, Camera, Image, Film, Home, Zap, Award, TrendingUp, Users, Clock, Star } from 'lucide-react'
import { motion } from 'framer-motion'
import './BookStudio.css'

const BookStudio = () => {
  const navigate = useNavigate()
  const [selectedType, setSelectedType] = useState('')
  const [selectedDate, setSelectedDate] = useState('')
  const [isSubmitted, setIsSubmitted] = useState(false)

  const studioTypes = [
    { 
      id: 'backdrop', 
      name: 'Студия с фоном', 
      icon: Image,
      color: '#FF6B35',
      description: 'Профессиональные фоны и освещение',
      gradient: 'linear-gradient(135deg, #FF6B35 0%, #F7931E 100%)',
      price: 'от 2500₽/час',
      area: '50 м²'
    },
    { 
      id: 'chromakey', 
      name: 'Хромакей', 
      icon: Film,
      color: '#F7931E',
      description: 'Зеленая стена для видеосъемки',
      gradient: 'linear-gradient(135deg, #F7931E 0%, #FF6B35 100%)',
      price: 'от 3000₽/час',
      area: '60 м²'
    },
    { 
      id: 'interior', 
      name: 'Интерьер', 
      icon: Home,
      color: '#FF8C42',
      description: 'Уютные интерьерные пространства',
      gradient: 'linear-gradient(135deg, #FF8C42 0%, #FF6B35 100%)',
      price: 'от 2800₽/час',
      area: '45 м²'
    },
    { 
      id: 'openspace', 
      name: 'Open Space', 
      icon: Zap,
      color: '#FF6B35',
      description: 'Большое открытое пространство',
      gradient: 'linear-gradient(135deg, #FF6B35 0%, #F7931E 100%)',
      price: 'от 3500₽/час',
      area: '80 м²'
    }
  ]

  const studios = [
    { name: 'StudioPro Premium', location: 'Центр', rating: 5, types: 'Все типы', capacity: 'VIP-студии' },
    { name: 'StudioPro Chroma', location: 'Тверская', rating: 5, types: 'Хромакей, Фон', capacity: '6 студий' },
    { name: 'StudioPro Interior', location: 'Арбат', rating: 5, types: 'Интерьер, Open Space', capacity: '4 студии' },
    { name: 'StudioPro Creative', location: 'Сокольники', rating: 5, types: 'Все типы', capacity: '8 студий' }
  ]

  const advantages = [
    { icon: Award, title: 'Профессиональное оборудование', text: 'Только лучшая техника' },
    { icon: Zap, title: 'Быстрое бронирование', text: 'Забронируйте за минуты' },
    { icon: Clock, title: 'Гибкий график', text: 'Работаем 24/7' },
    { icon: TrendingUp, title: 'Высокий рейтинг', text: '98% довольных клиентов' }
  ]

  const handleSubmit = (e) => {
    e.preventDefault()
    if (selectedType && selectedDate) {
      const savedBookings = localStorage.getItem('studioBookings')
      const bookings = savedBookings ? JSON.parse(savedBookings) : []
      
      const typeData = studioTypes.find(t => t.id === selectedType)
      const randomStudio = studios[Math.floor(Math.random() * studios.length)]
      
      const newBooking = {
        id: Date.now(),
        studio: randomStudio.name,
        date: selectedDate,
        equipment: typeData.description,
        review: ''
      }

      bookings.push(newBooking)
      localStorage.setItem('studioBookings', JSON.stringify(bookings))
      
      setIsSubmitted(true)
      setTimeout(() => {
        setIsSubmitted(false)
        setSelectedType('')
        setSelectedDate('')
        navigate('/')
      }, 2000)
    }
  }

  const minDate = new Date().toISOString().split('T')[0]
  const maxDate = new Date(Date.now() + 90 * 24 * 60 * 60 * 1000).toISOString().split('T')[0]

  return (
    <div className="book-studio-page">
      <div className="container">
        <motion.section 
          className="book-hero"
          initial={{ opacity: 0, y: 30 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6 }}
        >
          <motion.div 
            className="book-icon-wrapper"
            initial={{ scale: 0, rotate: 180 }}
            animate={{ scale: 1, rotate: 0 }}
            transition={{ type: "spring", stiffness: 200, damping: 15, delay: 0.3 }}
          >
            <Camera size={96} strokeWidth={2.5} className="book-hero-icon" />
          </motion.div>
          <h1 className="book-title">Забронировать студию</h1>
          <p className="book-subtitle">
            Выберите тип студии и удобную дату для вашего творческого проекта
          </p>
        </motion.section>

        <motion.section 
          className="booking-section"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.2 }}
        >
          <div className="booking-form-wrapper">
            <form className="booking-form" onSubmit={handleSubmit}>
              <div className="form-group">
                <label className="form-label">
                  <Camera size={32} strokeWidth={2.5} className="label-icon" />
                  Выберите тип студии
                </label>
                <div className="types-grid">
                  {studioTypes.map((type) => {
                    const IconComponent = type.icon
                    return (
                      <motion.button
                        key={type.id}
                        type="button"
                        className={`type-card ${selectedType === type.id ? 'selected' : ''}`}
                        onClick={() => setSelectedType(type.id)}
                        whileHover={{ y: -4, transition: { duration: 0.2 } }}
                        whileTap={{ scale: 0.98 }}
                        style={{ 
                          borderColor: selectedType === type.id ? type.color : 'rgba(255, 107, 53, 0.4)',
                          background: selectedType === type.id ? (type.color + '20') : 'rgba(26, 26, 26, 0.95)'
                        }}
                      >
                        <div className="type-icon-wrapper" style={{ background: type.color + '30' }}>
                          <IconComponent size={52} strokeWidth={2.5} color={type.color} />
                        </div>
                        <div className="type-content">
                          <h3 className="type-name">{type.name}</h3>
                          <p className="type-description">{type.description}</p>
                          <div className="type-meta">
                            <span className="type-area">{type.area}</span>
                            <span className="type-price">{type.price}</span>
                          </div>
                        </div>
                        {selectedType === type.id && (
                          <motion.div 
                            className="type-check"
                            initial={{ scale: 0 }}
                            animate={{ scale: 1 }}
                            style={{ background: type.gradient }}
                          >
                            <CheckCircle size={32} strokeWidth={2.5} color="white" />
                          </motion.div>
                        )}
                      </motion.button>
                    )
                  })}
                </div>
              </div>

              <div className="form-group">
                <label htmlFor="date" className="form-label">
                  <Calendar size={32} strokeWidth={2.5} className="label-icon" />
                  Дата бронирования
                </label>
                <input
                  type="date"
                  id="date"
                  className="form-input"
                  value={selectedDate}
                  onChange={(e) => setSelectedDate(e.target.value)}
                  min={minDate}
                  max={maxDate}
                  required
                />
              </div>

              <motion.button 
                type="submit" 
                className="submit-button" 
                disabled={!selectedType || !selectedDate}
                whileHover={{ scale: 1.03, y: -4 }}
                whileTap={{ scale: 0.98 }}
              >
                <span>Забронировать студию</span>
                <CheckCircle size={32} strokeWidth={2.5} className="btn-icon" />
              </motion.button>

              {isSubmitted && (
                <motion.div 
                  className="success-message"
                  initial={{ opacity: 0, scale: 0.9 }}
                  animate={{ opacity: 1, scale: 1 }}
                  transition={{ duration: 0.3 }}
                >
                  <CheckCircle size={40} strokeWidth={2.5} className="success-icon" />
                  <span>Вы успешно забронировали студию!</span>
                </motion.div>
              )}
            </form>
          </div>
        </motion.section>

        <motion.section 
          className="advantages-section"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.4 }}
        >
          <h2 className="section-title">Наши преимущества</h2>
          <div className="advantages-grid">
            {advantages.map((advantage, index) => {
              const IconComponent = advantage.icon
              return (
                <motion.div
                  key={advantage.title}
                  className="advantage-card"
                  initial={{ opacity: 0, y: 20 }}
                  animate={{ opacity: 1, y: 0 }}
                  transition={{ delay: index * 0.1, duration: 0.5 }}
                  whileHover={{ y: -4, transition: { duration: 0.2 } }}
                >
                  <div className="advantage-icon">
                    <IconComponent size={44} strokeWidth={2.5} color="#FF6B35" />
                  </div>
                  <h3 className="advantage-title">{advantage.title}</h3>
                  <p className="advantage-text">{advantage.text}</p>
                </motion.div>
              )
            })}
          </div>
        </motion.section>

        <motion.section 
          className="studios-section"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.5 }}
        >
          <h2 className="section-title">Наши студии</h2>
          <div className="studios-grid">
            {studios.map((studio, index) => (
              <motion.div
                key={studio.name}
                className="studio-card"
                initial={{ opacity: 0, scale: 0.9 }}
                animate={{ opacity: 1, scale: 1 }}
                transition={{ delay: index * 0.1, duration: 0.5 }}
                whileHover={{ y: -4, transition: { duration: 0.2 } }}
              >
                <div className="studio-avatar">
                  <Camera size={44} strokeWidth={2.5} />
                </div>
                <h3 className="studio-name">{studio.name}</h3>
                <p className="studio-location">{studio.location}</p>
                <div className="studio-rating">
                  {Array.from({ length: 5 }, (_, i) => (
                    <Star key={i} size={18} fill="#F59E0B" stroke="#F59E0B" strokeWidth={2} />
                  ))}
                </div>
                <p className="studio-types">{studio.types}</p>
                <div className="studio-capacity">
                  <Users size={18} strokeWidth={2} />
                  <span>{studio.capacity}</span>
                </div>
              </motion.div>
            ))}
          </div>
        </motion.section>
      </div>
    </div>
  )
}

export default BookStudio

